*! varbench -- compare VAR forecast RMSEs to alternative forecast methods
*! version 1.0		Sean Becketti		January 2012
program define varbench
version 12.0
quietly {
	syntax varlist [, End(string) Lags(numlist ascending) Prefix(string) STArt(string) Steps(numlist ascending) ]
	tsset
	local date "`r(timevar)'"
	local unit "`r(unit)'"
	local fmt : format `date'
	if ("`lags'"=="") {	Default order is 2
		local lags "1 2"
	}
	local i : word count `lags'
	local order : word `i' of `lags'
	if ("`steps'"=="") {	Default steps ahead
		if ("`unit'"=="monthly") {
			local steps "3 6 9 12"
		}
		else {
			local steps "2 4 8"	/* quarterly? */
		}
	}
	local i : word count `steps'
	local maxstep : word `i' of `steps'
	local nvars : word count `varlist'
	local rows : word count `steps'
/*
	Set minimum and maximum estimation samples
*/
	if ("`start'"=="" ) {
		local start = _N/2
	}
	else {
		count if `date' <= `start'
		local start = `r(N)'
	}
	local start = max(`start',5*`nvars'*`order')
	if ("`end'"=="") {
		local end = _N - `maxstep'
	}
	else {
		count if `date' <= `end'
		local end = `r(N)'
	}
	if (`end' - `start' < 10) {
		noi di as error "Not enough observations"
		error 98
	}
/*
	Generate temporary variables to store the projections and squared errors
*/
	tempvar f e
	gen `f' = .	/* forecasts */
	gen `e' = .	/* squared errors */
/*
	First: Mean
*/
	foreach v of varlist `varlist' {
		forvalues i = `start'/`end' {
			sum `v' in f/`i'
			replace `f' = `r(mean)' in `i'
		}
		foreach n in `steps' {
			replace `e' = (`v'[_n +`n'] - `f')^2 in `start'/`end'
			sum `e'
			local mn`n'`v' = sqrt(`r(mean)')
		}
	}
/*
	Second: Random walk
*/
	replace `e' = .
	foreach v of varlist `varlist' {
		foreach n in `steps' {
			replace `e' = (`v'[_n+`n'] - `v')^2 in `start'/`end'
			summarize `e'
			local rw`n'`v' = sqrt(`r(mean)')
		}
	}
/*
	Third: Univariate AR
*/
	foreach v of varlist `varlist' {	/* tempvars for AR and VAR */
		foreach n in `steps' {
			tempvar e`n'`v'
			gen `e`n'`v'' = .
		}
	}
	forvalues i = `start'/`end' {
		foreach v of varlist `varlist' {
			arima `v' in f/`i', ar(`lags')
			capture drop `f'
			predict `f', dynamic(`i')
			foreach n in `steps' {
				replace `e`n'`v'' = (`v'[`i' + `n'] - `f'[`i' + `n'])^2 in `i'
			}
		}
	}
	foreach v of varlist `varlist' {
		foreach n in `steps' {
			summarize `e`n'`v''
			local ar`n'`v' = sqrt(`r(mean)')
		}
	}
/*
	Third: VAR

	First find a prefix that won't cause a collision with existing variables
*/
	if ("`prefix'"!="") {
		local pfx "`prefix'"
	}
	else{
		local prefix "ZYXWVUTSRQ"
		local len = length("`prefix'")
		local i 1
		local bad 1
		while ((`i' < `len') & `bad') {
			local pfx = substr("`prefix'",`i',1) + "_"
			local bad 0
			foreach v of varlist `varlist' {
				cap conf new v `pfx'`v'
				local bad = `bad' & _rc
			}
		}
		if (`bad') {
			noi di as error "Unable to find legal prefix for fcast!"
			error 98
		}
	}
	foreach v of varlist `varlist' {	/* tempvars for AR and VAR */
		foreach n in `steps' {
			replace `e`n'`v'' = .
		}
	}
	forvalues i = `start'/`end' {
		var `varlist' in f/`i', lags(`lags')
		fcast compute `pfx', step(`maxstep') nose replace
		foreach v of varlist `varlist' {
			foreach n in `steps' {
 				replace `e`n'`v'' = (`v'[`i' + `n'] - `pfx'`v'[`i' + `n'])^2 in `i'
			}
		}
	}
	foreach v of varlist `varlist' {
		drop `pfx'`v'			/* Drop the VAR forecast */
		foreach n in `steps' {
			summarize `e`n'`v''
			local var`n'`v' = sqrt(`r(mean)')
		}
	}
/*
	Display results
*/
	noi di _n as text "RMSE of simulated out-of-sample forecasts"
	noi di as text "End of sample: " as result `fmt' = `date'[`start'] " to " `fmt' = `date'[`end'] _n
	foreach v of varlist `varlist' {
		local vl : variable label `v'
		if ("`vl'"=="") {
			local vl "`v'"
		}
		noi di as text "`vl'" 
		noi di _s(20) as text "Method" _s(21) "% Improvement"
		noi di _s(9) "----------------------------" _s(7) "--------------------"
		noi di as text "Horizon" _s(2) "Mean" _s(5) "RW" _s(6) "AR" _s(4) "VAR" /*
*/        		_s(9) "Mean" _s(5) "RW" _s(6) "AR"
		foreach n in `steps' {
			noi di as text %4.0f = `n' _s(3) as result %6.2f = `mn`n'`v'' 	/*
*/				_s(2) as result %6.2f = `rw`n'`v''			/*
*/				_s(2) as result %6.2f = `ar`n'`v''			/*
*/				_s(2) as result %6.2f = `var`n'`v'' _continue
local pim = 100*(1-(`var`n'`v''/`mn`n'`v''))
local prw = 100*(1-(`var`n'`v''/`rw`n'`v''))
local par = 100*(1-(`var`n'`v''/`ar`n'`v''))
			noi di _s(4) as result %6.0f = `pim' 	/*
*/				_s(3) as result %6.0f = `prw'			/*
*/				_s(2) as result %6.0f = `par'

		}
		noi di _n
	}



}
end
